'
' =============================================================================
' DHL All-in-One Application - SAP TRP File Access Script
' Author: Dario Pascoal
'
' Description: This VBScript automates the process of accessing and opening TRP
' (Tool Registration Package) files from the SAP ZPACK transaction within the
' DHL All-in-One application workflow. It provides automated navigation through
' SAP GUI to retrieve technical documentation and visual materials for equipment
' inspection and quality control processes.
'
' Primary Functions:
' 1. SAP GUI Security Configuration - Disable scripting warnings for automation
' 2. ZPACK Transaction Navigation - Automated access to material documentation
' 3. File Search and Retrieval - Intelligent search for PICT and DPI files
' 4. Document Opening Automation - Direct access to technical documentation
' 5. Error Handling and Logging - Comprehensive logging for troubleshooting
' 6. Status Reporting - Clear communication of operation results
'
' Key Features:
' - Automated SAP GUI security dialog handling
' - Priority-based file search (PICT files preferred over DPI)
' - Comprehensive error handling with detailed logging
' - Command-line argument processing for material numbers
' - Registry modification for seamless SAP GUI automation
' - Intelligent file type detection and handling
' - Professional logging with timestamps and context
'
' Script Workflow:
' 1. Initialize Environment: Set up SAP GUI scripting and disable security warnings
' 2. Process Arguments: Extract material number and optional log file path
' 3. Connect to SAP: Establish connection to active SAP GUI session
' 4. Navigate to ZPACK: Open the material package transaction
' 5. Search Materials: Enter material number and execute search
' 6. Analyze Results: Scan file list for PICT or DPI documents
' 7. Open Documents: Launch appropriate documentation viewer
' 8. Report Status: Return success/failure status with details
'
' Technical Implementation:
' - VBScript with COM object automation for SAP GUI interaction
' - Windows Registry manipulation for security configuration
' - File system operations for logging and error handling
' - Error handling with On Error Resume Next patterns
' - Object-oriented approach with proper cleanup procedures
' - Command-line interface with argument validation
'
' Integration Points:
' - SAP GUI Scripting Engine for automated navigation
' - Windows Shell for dialog handling and security management
' - File System Object for logging and file operations
' - DHL All-in-One main application for status reporting
' - SAP ZPACK transaction for material documentation access
'
' Security Features:
' - Temporary registry modifications with automatic cleanup
' - Error handling to prevent script crashes and data loss
' - Logging for audit trails and troubleshooting
' - Secure SAP GUI connection handling
' - Proper object cleanup to prevent memory leaks
'
' Error Handling:
' - Comprehensive error catching with detailed error messages
' - Graceful degradation when optional features fail
' - Detailed logging for debugging and maintenance
' - Status codes for integration with calling applications
' - Automatic cleanup procedures on script termination
' =============================================================================

' ============================================================================
' GLOBAL ERROR HANDLING CONFIGURATION
' ============================================================================
' Enable error handling to prevent script crashes and provide graceful
' degradation when encountering unexpected conditions or SAP GUI issues
On Error Resume Next

' ============================================================================
' ENVIRONMENT INITIALIZATION
' ============================================================================
' Initialize Windows Script Host shell object for system interactions
' This object handles registry modifications and security dialog automation
Set WSHShell = CreateObject("WScript.Shell")

' ============================================================================
' SAP GUI SECURITY CONFIGURATION
' ============================================================================
' Disable SAP GUI scripting security warnings to enable automated interaction
' These registry modifications are essential for unattended script execution
' and prevent user prompts that would interrupt the automation workflow

' Disable attachment warnings when scripts connect to SAP GUI sessions
' This prevents popup dialogs that would halt automation processes
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"

' Disable connection warnings for script-to-SAP communication
' This ensures seamless connection establishment without user intervention
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"

' Disable allow list requirement warnings for automated scripting
' This prevents security prompts that would require manual user interaction
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"

' Disable allow list usage to prevent script blocking
' This allows our automation script to execute without whitelist restrictions
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' ============================================================================
' COMMAND LINE ARGUMENT PROCESSING
' ============================================================================
' Extract and validate command line arguments for material number and logging
' The script requires a material number and optionally accepts a log file path
' for detailed operation tracking and troubleshooting capabilities

' Get material number from first command line argument (required)
' This material number identifies the equipment for which TRP files are requested
materialNumber = WScript.Arguments.Item(0)

' ============================================================================
' LOGGING SYSTEM INITIALIZATION
' ============================================================================
' Setup comprehensive logging system for operation tracking and debugging
' The logging system supports detailed file logging with timestamps for
' audit trails, troubleshooting, and operational monitoring

' Declare log file objects and path variables
' These objects handle all file-based logging operations
Dim logFile, logFilePath

' Initialize logging system if log file path is provided as second argument
' Enhanced logging provides detailed operation tracking for debugging
If WScript.Arguments.Count > 1 Then
    logFilePath = WScript.Arguments.Item(1)
    
    ' Attempt to open log file in append mode for continuous logging
    ' Append mode (8) preserves existing log entries and adds new ones
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    ' Handle log file opening errors gracefully
    ' If logging fails, continue script execution without logging
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Log script startup information for audit trail
        ' This creates a clear session marker in the log file
        LogToFile "========================================="
        LogToFile "Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Material Number: " & materialNumber
        LogToFile "========================================="
    End If
End If

' ============================================================================
' LOGGING UTILITY FUNCTIONS
' ============================================================================
' Centralized logging functions to ensure consistent operation tracking
' and error reporting across all script operations and SAP interactions

' File-only logging function for detailed internal operations
' Used for technical details that don't need console output
Sub LogToFile(message)
    On Error Resume Next
    ' Only write to file if logging is enabled and file is available
    If Not logFile Is Nothing Then
        ' Format: timestamp - [script name] - message for easy parsing
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

' Dual logging function for important messages
' Outputs to both console (for immediate feedback) and log file (for records)
Sub LogMessage(message)
    ' Display message to user for immediate feedback
    WScript.Echo message
    ' Also save to log file for permanent record
    LogToFile message
End Sub

' ============================================================================
' SAP GUI CONNECTION ESTABLISHMENT
' ============================================================================
' Establish connection to SAP GUI application and active session
' This section handles the complex process of connecting to SAP through
' the SAP GUI Scripting Engine and manages connection error handling

' Initialize SAP GUI application object if not already connected
' The application object is the top-level SAP GUI automation interface
If Not IsObject(application) Then
   ' Connect to the SAP GUI Automation Server
   ' This server manages all SAP GUI automation and scripting operations
   Set SapGuiAuto  = GetObject("SAPGUI")
   ' Get the scripting engine from the SAP GUI server
   ' The scripting engine provides programmatic access to SAP GUI elements
   Set application = SapGuiAuto.GetScriptingEngine
End If

' Initialize SAP connection object if not already established
' The connection object represents a specific SAP system connection
If Not IsObject(connection) Then
   ' Get the first (primary) connection from available connections
   ' Children(0) accesses the first active connection to SAP
   Set connection = application.Children(0)
End If

' Initialize SAP session object for user interaction simulation
' The session object provides access to SAP screens and transactions
If Not IsObject(session) Then
   ' Get the first (primary) session from the active connection
   ' This session will be used for all SAP GUI automation commands
   Set session = connection.Children(0)
End If

' Connect VBScript event handlers to SAP GUI objects
' This enables the script to respond to SAP GUI events and errors
If IsObject(WScript) Then
   ' Connect session events for user interaction monitoring
   ' "on" prefix enables event handling for session-level events
   WScript.ConnectObject session,     "on"
   ' Connect application events for system-level monitoring
   ' This handles application-wide events and error conditions
   WScript.ConnectObject application, "on"
End If

' ============================================================================
' SECURITY DIALOG AUTOMATION
' ============================================================================
' Automated handling of SAP GUI security dialogs that may appear during
' script execution, ensuring unattended operation without user intervention

' Handle security dialog automation for unattended script execution
' This function automatically dismisses security warnings that could
' interrupt the automation workflow and require manual intervention
Sub HandleSecurityDialog()
    Dim shell, timeout, dialogFound
    Set shell = CreateObject("WScript.Shell")
    timeout = 0
    dialogFound = False
    
    ' Wait up to 10 seconds for security dialog to appear
    ' This prevents infinite waiting while allowing time for dialog display
    Do While timeout < 10
        ' Check if SAP GUI Security dialog is active
        ' AppActivate returns True if the specified window title is found
        If shell.AppActivate("SAP GUI Security") Then
            ' Brief pause to ensure dialog is fully loaded
            ' This prevents sending keys before dialog is ready
            WScript.Sleep 500
            
            ' Press space to check "Remember my decision" checkbox
            ' This prevents the same security dialog from appearing repeatedly
            shell.SendKeys " "
            
            ' Brief pause between key operations for reliability
            ' Ensures the checkbox state change is processed
            WScript.Sleep 200
            
            ' Press Enter to click the "Allow" button
            ' This grants permission for the script to continue execution
            shell.SendKeys "{ENTER}"
            dialogFound = True
            ' Exit loop immediately after successful dialog handling
            Exit Do
        End If
        
        ' Wait 500ms before checking again to avoid excessive CPU usage
        ' This provides a reasonable balance between responsiveness and performance
        WScript.Sleep 500
        timeout = timeout + 1
    Loop
End Sub

' ============================================================================
' MAIN SAP AUTOMATION WORKFLOW
' ============================================================================
' Execute the complete TRP file search and opening workflow
' This section orchestrates all SAP GUI interactions to locate and open
' the appropriate technical documentation for the specified material

' Execute security dialog handling before starting SAP operations
' This ensures any security prompts are automatically dismissed
HandleSecurityDialog()

' ============================================================================
' SAP TRANSACTION NAVIGATION
' ============================================================================
' Navigate to ZPACK transaction for material documentation access
' ZPACK is the custom SAP transaction for accessing material packaging
' and technical documentation files

' Maximize SAP GUI window to ensure all elements are visible
' This prevents issues with hidden or partially visible UI elements
session.findById("wnd[0]").maximize

' Navigate to ZPACK transaction using the command field
' The transaction code directs SAP to the material packaging interface
session.findById("wnd[0]/tbar[0]/okcd").text = "zpack"

' Execute the transaction command (equivalent to pressing Enter)
' sendVKey 0 sends the Enter key to execute the transaction
session.findById("wnd[0]").sendVKey 0

' ============================================================================
' MATERIAL SEARCH EXECUTION
' ============================================================================
' Input material number and execute search to locate documentation files
' This section handles the material search process within ZPACK transaction

' Enter the material number in the material search field
' S_MATNR-LOW is the standard SAP field for material number input
session.findById("wnd[0]/usr/ctxtS_MATNR-LOW").text = materialNumber

' Execute the search using function key F8 (Execute)
' sendVKey 8 corresponds to F8 which triggers the search operation
session.findById("wnd[0]").sendVKey 8

' ============================================================================
' DOCUMENT ANALYSIS AND SELECTION
' ============================================================================
' Analyze the search results to locate appropriate TRP documentation files
' This section scans through available files and prioritizes document types
' based on DHL requirements and technical specification standards

' Get reference to the file list table in the SAP GUI
' This table contains all available documentation files for the material
Set fileTable = session.findById("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3/ssubTAB_9100_SCA:ZDGP_MATERIAL_OVERVIEW:9103/cntlCONTROL_9103_2/shellcont/shell")

' Initialize file search status flags
' These flags track whether appropriate documentation has been found
found = False        ' General file found flag
foundDPI = False     ' Specific flag for DPI (high-resolution) files
rowIndex = -1        ' Row index for found PICT file
dpiRowIndex = -1     ' Row index for found DPI file
foundFileName = ""   ' Name of the found file

' ============================================================================
' FILE LIST SCANNING AND PRIORITIZATION
' ============================================================================
' Scan through available files to locate the most appropriate documentation
' Priority order: PICT files (preferred) -> DPI files (acceptable alternative)
' This ensures the best quality documentation is selected when available

' Iterate through all rows in the file table to examine available documents
' The loop checks each file name against our priority criteria
For i = 0 To fileTable.RowCount - 1
    ' Extract the file name from the document number column
    ' DOKNR column contains the document identifier/filename
    fileName = fileTable.GetCellValue(i, "DOKNR")
    
    ' Check for PICT files (highest priority)
    ' PICT files contain pictorial/graphical technical documentation
    If InStr(1, fileName, "PICT", vbTextCompare) > 0 Then
        found = True
        rowIndex = i
        foundFileName = fileName
        ' Exit immediately when PICT file is found (highest priority)
        Exit For
    ' Check for DPI files (secondary priority)
    ' DPI files contain high-resolution technical drawings
    ElseIf InStr(1, fileName, "DPI", vbTextCompare) > 0 Then
        foundDPI = True
        dpiRowIndex = i
        ' Continue scanning - don't exit yet in case PICT file exists
    End If
Next

' ============================================================================
' DOCUMENT OPENING AND RESULT PROCESSING
' ============================================================================
' Open the selected document and provide feedback on the operation results
' This section handles both successful document opening and error conditions

' Process search results based on file availability and priority
If found Then
    ' PICT file found and selected (highest priority option)
    ' Set the current cell to the found PICT file row
    fileTable.setCurrentCell rowIndex, "DOKNR"
    ' Select the entire row containing the PICT file
    fileTable.selectedRows = rowIndex
    ' Double-click to open the document in the appropriate viewer
    fileTable.doubleClickCurrentCell
    
    ' Allow sufficient time for document viewer to launch and load
    ' 3-second delay ensures proper application startup
    WScript.Sleep 3000
    
    ' Log successful operation with filename for audit trail
    LogMessage "SUCCESS: " & foundFileName
    
    ' Note: Document remains open and SAP stays on document list screen
    ' This allows users to continue working with the opened documentation
    
Else
    ' No PICT file found, check for DPI alternative
    If foundDPI Then
        ' Return DPI found status - handling will be done in JavaScript
        LogMessage "DPI-FOUND"
Else
    ' No PICT file found, check for DPI alternative
    If foundDPI Then
        ' DPI file found as secondary option
        ' DPI files provide acceptable alternative when PICT files unavailable
        fileTable.setCurrentCell dpiRowIndex, "DOKNR"
        fileTable.selectedRows = dpiRowIndex
        fileTable.doubleClickCurrentCell
        
        ' Allow time for DPI document viewer to launch
        WScript.Sleep 3000
        
        ' Log successful DPI file opening
        LogMessage "SUCCESS: DPI file opened"
        
    Else
        ' No suitable documentation files found for this material
        ' This indicates either missing documentation or incorrect material number
        LogMessage "NOT-FOUND"
    End If
End If

' ============================================================================
' CLEANUP AND FINALIZATION
' ============================================================================
' Proper cleanup procedures to ensure resources are released and logging
' is finalized correctly before script termination

' Cleanup function to properly close log file and release resources
' This function ensures proper log file closure and session cleanup
Sub CloseLogFile()
    On Error Resume Next
    ' Check if log file is still open and available
    If Not logFile Is Nothing Then
        ' Write session end marker to log for clear audit trail
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        ' Properly close the log file to ensure all data is written
        logFile.Close
        ' Release the log file object to free system resources
        Set logFile = Nothing
    End If
End Sub

' Execute cleanup procedures before script termination
' This ensures proper resource cleanup and log file closure
CloseLogFile()